import os
import struct
import zlib
import pickle
import hashlib
import hmac
from typing import Dict

from Crypto.Cipher import AES


MAGIC = b"SDA1"
VERSION = 1

SECRET_KEY = (
    b"\x4B\xE2\xF5\xE5\x64\x5C\x4D\xB7\x12\x83\xBC\x30\x3E\x27\x27\x43"
    b"\xAA\xD2\x66\xCA\x78\x90\x45\x58\x46\x42\x69\xBC\x0D\x9E\xD0\x0C"
)

_ARCHIVE_KEYS: Dict[str, dict] = {}


def _derive_keys(salt: bytes):
    base = SECRET_KEY + salt
    enc_key = hashlib.sha256(b"ENC" + base).digest()  # 32 байта
    mac_key = hashlib.sha256(b"MAC" + base).digest()  # 32 байта
    return enc_key, mac_key


def _aes_gcm_encrypt(enc_key: bytes, nonce: bytes, data: bytes):
    cipher = AES.new(enc_key, AES.MODE_GCM, nonce=nonce)
    ct, tag = cipher.encrypt_and_digest(data)
    return ct, tag


def _aes_gcm_decrypt(enc_key: bytes, nonce: bytes, data: bytes, tag: bytes):
    cipher = AES.new(enc_key, AES.MODE_GCM, nonce=nonce)
    return cipher.decrypt_and_verify(data, tag)


def build_archive(output_path, files):
    salt = os.urandom(16)
    enc_key, mac_key = _derive_keys(salt)

    header_size = 4 + 1 + 16 + 8 + 8 + 12 + 16

    index = {}

    with open(output_path, "wb+") as out:
        out.write(MAGIC)
        out.write(struct.pack("<B", VERSION))
        out.write(salt)
        out.write(struct.pack("<Q", 0))
        out.write(struct.pack("<Q", 0))
        out.write(b"\x00" * 12)
        out.write(b"\x00" * 16)

        assert out.tell() == header_size

        offset = header_size

        for logical, real in files:
            with open(real, "rb") as f:
                raw = f.read()

            nonce = os.urandom(12)
            ct, tag = _aes_gcm_encrypt(enc_key, nonce, raw)

            out.write(ct)
            length = len(ct)

            index[logical] = [(offset, length, nonce, tag)]
            offset += length

        index_offset = offset

        index_bytes = pickle.dumps(index, protocol=4)
        comp = zlib.compress(index_bytes)

        index_nonce = os.urandom(12)
        index_ct, index_tag = _aes_gcm_encrypt(enc_key, index_nonce, comp)
        index_length = len(index_ct)

        out.write(index_ct)

        out.seek(4 + 1 + 16, os.SEEK_SET)
        out.write(struct.pack("<Q", index_offset))
        out.write(struct.pack("<Q", index_length))
        out.write(index_nonce)
        out.write(index_tag)

        out.flush()
        out.seek(0, os.SEEK_END)
        total_len = out.tell()
        out.seek(0)
        content = out.read(total_len)

        h = hmac.new(mac_key, content, hashlib.sha256).digest()

        out.seek(0, os.SEEK_END)
        out.write(h)


def read_index(infile):
    fn = getattr(infile, "name", None)
    if not fn:
        raise Exception("No filename for archive file object")

    fn = os.path.abspath(fn)

    with open(fn, "rb") as f:
        magic = f.read(4)
        if magic != MAGIC:
            raise Exception("Not SDA archive")

        ver = struct.unpack("<B", f.read(1))[0]
        if ver != VERSION:
            raise Exception("Unsupported SDA version")

        salt = f.read(16)
        enc_key, mac_key = _derive_keys(salt)

        index_offset = struct.unpack("<Q", f.read(8))[0]
        index_length = struct.unpack("<Q", f.read(8))[0]
        index_nonce = f.read(12)
        index_tag = f.read(16)

        f.seek(0, os.SEEK_END)
        total_len = f.tell()
        if total_len < 32:
            raise Exception("Archive too small for HMAC")

        content_len = total_len - 32
        f.seek(0)
        content = f.read(content_len)
        expected_hmac = f.read(32)

        calc_hmac = hmac.new(mac_key, content, hashlib.sha256).digest()
        if not hmac.compare_digest(expected_hmac, calc_hmac):
            raise Exception("Archive HMAC mismatch")

        f.seek(index_offset)
        index_ct = f.read(index_length)
        comp = _aes_gcm_decrypt(enc_key, index_nonce, index_ct, index_tag)
        index_bytes = zlib.decompress(comp)
        index = pickle.loads(index_bytes)

    _ARCHIVE_KEYS[fn] = {
        "enc_key": enc_key,
    }

    return index


def decrypt_file_chunk(archive_path: str, name: str,
                       offset: int, length: int,
                       nonce: bytes, tag: bytes) -> bytes:
    ap = os.path.abspath(archive_path)
    info = _ARCHIVE_KEYS.get(ap)
    if info is None:
        raise Exception("Archive key not initialized for %r" % ap)

    enc_key = info["enc_key"]

    with open(ap, "rb") as f:
        f.seek(offset)
        ct = f.read(length)

    pt = _aes_gcm_decrypt(enc_key, nonce, ct, tag)
    return pt
